<?php
/**
 *  ╔═══════════════════════════════════════════════════╗
 *  ║                                                   ║
 *  ║     ██╗  ██╗   █████╗    ██████╗                  ║
 *  ║     ██║  ██║  ██╔══██╗  ██╔═══██╗                 ║
 *  ║     ███████║  ███████║  ██║   ██║                 ║
 *  ║     ██╔══██║  ██╔══██║  ██║   ██║                 ║
 *  ║     ██║  ██║  ██║  ██║  ╚██████╔╝   SNS           ║
 *  ║                                                   ║    
 *  ║                                                   ║    
 *  ║     © 2023 HaoSNS™ All Rights Reserved            ║
 *  ║     官方网站: https://www.haosns.com                *
 *  ║     本代码由赣州乐易网络科技有限公司®提供             *
 *  ║                                                    *
 *  ║   未经授权禁止复制、传播或用于其他商业目的            *
 *  ║                                                   ║
 *  ╚═══════════════════════════════════════════════════╝
 */



namespace app\api\validate;


use app\common\cache\UserAccountSafeCache;
use app\common\enum\LoginEnum;
use app\common\enum\UserTerminalEnum;
use app\common\enum\YesNoEnum;
use app\common\model\User;
use app\common\service\ConfigService;
use app\common\service\sms\SmsDriver;
use app\common\validate\BaseValidate;
use think\facade\Config;

class LoginValidate extends BaseValidate
{
    protected $rule = [
        'terminal' => 'require|in:' . UserTerminalEnum::WECHAT_MMP . ',' . UserTerminalEnum::WECHAT_OA . ','
            . UserTerminalEnum::H5 . ',' . UserTerminalEnum::PC . ','
            . UserTerminalEnum::IOS . ',' . UserTerminalEnum::ANDROID ,
        'scene' => 'require|in:1,2|checkConfig',
        // 'account' => 'require|string', // 可能是这里的正则验证失败
        // 'mobile' => 'require|string',
        'code' => 'number'
    ];

    protected $message = [
        'terminal.require' => '终端参数缺失',
        'terminal.in' => '终端参数状态值不正确',
        'scene.require' => '场景不能为空',
        'scene.in' => '场景值错误',
        // 'account.require' => '请输入账号',
        'password.require' => '请输入密码',
        'mobile.require' => '请输入手机号',
        // 'mobile.mobile' => '无效的手机号',
        // 'account.mobile' => '无效的手机号',
    ];

    /**
     * @notes 账号密码/手机号密码/手机号验证码登录场景
     * @return LoginValidate
     */
    public function sceneAccount()
    {
        // return $this->remove('mobile', 'require|mobile')->append('account','mobile');
    }

    /**
     * @notes 发送登录验证码
     * @return LoginValidate
     */
    public function sceneCaptcha()
    {
        return $this->only(['mobile']);
    }

    /**
     * @notes 密码验证
     * @param $password
     * @param $other
     * @param $data
     * @return bool|string
     * @throws \think\db\exception\DataNotFoundException
     * @throws \think\db\exception\DbException
     * @throws \think\db\exception\ModelNotFoundException
     */
    public function checkPassword($password, $other, $data)
    {
        //后台账号安全机制，连续输错后锁定，防止账号密码暴力破解
        $userAccountSafeCache = new UserAccountSafeCache();
        if (!$userAccountSafeCache->isSafe()) {
            return '密码连续' . $userAccountSafeCache->count . '次输入错误，请' . $userAccountSafeCache->minute . '分钟后重试';
        }

        $where = [];

        if($data['scene'] == LoginEnum::MOBILE_PASSWORD || $data['scene'] == LoginEnum::MOBILE_CAPTCHA) {
            // 手机号密码登录
            $where = ['mobile' => $data['account']];
        }

        // dd($where);

        $userInfo = User::where($where)
            ->whereOr('account',$data['account'])
            
            ->field(['password,disable,salt'])
            ->find();

        if (empty($userInfo)) {
            $userInfo = User::where($where)
            ->whereOr('email',$data['account'])
            
            ->field(['password,disable,salt'])
            ->find();
        }

        if (empty($userInfo)) {
            return '用户不存在';
        }

        if ($userInfo['disable'] === YesNoEnum::YES) {
            return '账号被冻结，请联系客服。';
        }

        if (empty($userInfo['password'])) {
            return '用户或者密码错误';
        }

        if ($userInfo['password'] != create_password($password, $userInfo['salt'])) {
            $userAccountSafeCache->record();
            return '密码错误';
        }
        $userAccountSafeCache->relieve();
        
        return true;
    }

    /**
     * @notes 校验登录设置
     * @return bool|string
     */
    public function checkConfig($scene, $rule, $data)
    {
        $config = ConfigService::get('config', 'login_way', []);
        
        if(!in_array($scene, $config)) {
            return '不支持的登录方式';
        }
        if(($scene == LoginEnum::MOBILE_PASSWORD) && !isset($data['password'])) {
            return '请输入密码';
        }
        if(($scene == LoginEnum::MOBILE_PASSWORD)) {
            return $this->checkPassword($data['password'], [], $data);
        }
        if($scene == LoginEnum::MOBILE_CAPTCHA && !isset($data['code'])) {
            return '请输入手机验证码';
        }
        if($scene == LoginEnum::MOBILE_CAPTCHA) {
            return $this->checkCode($data['code'], [], $data);
        }
        return true;
    }

    /**
     * @notes 校验验证码
     * @param $code
     * @param $rule
     * @param $data
     * @return bool|string
     */
    public function checkCode($code, $rule, $data)
    {
        $smsDriver = new SmsDriver();
        $result = $smsDriver->verify($data['account'], $code);
        if($result) {
            return true;
        }
        return '验证码错误';
    }
}