<?php
/**
 *  ╔═══════════════════════════════════════════════════╗
 *  ║                                                   ║
 *  ║     ██╗  ██╗   █████╗    ██████╗                  ║
 *  ║     ██║  ██║  ██╔══██╗  ██╔═══██╗                 ║
 *  ║     ███████║  ███████║  ██║   ██║                 ║
 *  ║     ██╔══██║  ██╔══██║  ██║   ██║                 ║
 *  ║     ██║  ██║  ██║  ██║  ╚██████╔╝   SNS           ║
 *  ║                                                   ║    
 *  ║                                                   ║    
 *  ║     © 2023 HaoSNS™ All Rights Reserved            ║
 *  ║     官方网站: https://www.haosns.com                *
 *  ║     本代码由赣州乐易网络科技有限公司®提供             *
 *  ║                                                    *
 *  ║   未经授权禁止复制、传播或用于其他商业目的            *
 *  ║                                                   ║
 *  ╚═══════════════════════════════════════════════════╝
 */


namespace app\api\validate;

use app\common\enum\WithdrawEnum;
use app\common\model\User;
use app\common\service\ConfigService;
use app\common\validate\BaseValidate;

/**
 * 提现验证器
 * Class WithdrawValidate
 * @package  app\api\validate
 */
class WithdrawValidate extends BaseValidate
{
    protected $rule = [
        'id' => 'require',
        'type' => 'require|in:1,2,3,4,5',
        'money' => 'require|gt:0|checkMoney',
        'account' => 'requireIf:type,3|requireIf:type,4|requireIf:type,5',
        'real_name' => 'requireIf:type,3|requireIf:type,4|requireIf:type,5|chs',
        'money_qr_code' => 'requireIf:type,4|requireIf:type,5',
        'bank' => 'requireIf:type,3',
        'subbank' => 'requireIf:type,3',
    ];

    protected $message = [
        'id.require' => '参数缺失',
        'type.require' => '请选择提现类型',
        'type.in' => '提现类型状态值错误',
        'money.require' => '请输入提现金额',
        'money.gt' => '提现金额须大于0',
        'account.requireIf' => '请输入提现账号',
        'real_name.requireIf' => '请输入真实姓名',
        'real_name.chs' => '真实姓名须为中文',
        'money_qr_code.requireIf' => '请上传收款码',
        'bank.requireIf' => '请输入提现银行',
        'subbank.requireIf' => '请输入银行支行',
    ];

    /**
     * @notes 提现申请场景
     * @return WithdrawValidate
     */
    public function sceneApply()
    {
        return $this->only(['type', 'money', 'account', 'real_name', 'money_qr_code', 'bank', 'subbank']);
    }

    /**
     * @notes 提现申请详情场景
     * @return WithdrawValidate
     */
    public function sceneDetail()
    {
        return $this->only(['id']);
    }


    /**
     * @notes 校验提现金额
     * @param $value
     * @param $rule
     * @param $data
     * @return bool|string
     */
    protected function checkMoney($value, $rule, $data)
    {
        // 可提现金额是否充足
        $user_earnings = User::where('id', $data['user_id'])->value('user_earnings');
        $user_earnings = is_null($user_earnings) ? 0 : $user_earnings;
        if ($value > $user_earnings){
            return '可提现金额不足';
        }

        // 最低提现金额
        $min_withdraw = ConfigService::get('config', 'withdraw_min_money', WithdrawEnum::DEFAULT_MIN_MONEY);
        if($value < $min_withdraw){
            return '最低提现'.$min_withdraw.'元';
        }

        // 最高提现金额
        $max_withdraw = ConfigService::get('config', 'withdraw_max_money', WithdrawEnum::DEFAULT_MAX_MONEY);
        if ($value > $max_withdraw){
            return '最高提现'.$max_withdraw.'元';
        }

        return true;
    }
}