<?php
/**
 *  ╔═══════════════════════════════════════════════════╗
 *  ║                                                   ║
 *  ║     ██╗  ██╗   █████╗    ██████╗                  ║
 *  ║     ██║  ██║  ██╔══██╗  ██╔═══██╗                 ║
 *  ║     ███████║  ███████║  ██║   ██║                 ║
 *  ║     ██╔══██║  ██╔══██║  ██║   ██║                 ║
 *  ║     ██║  ██║  ██║  ██║  ╚██████╔╝   SNS           ║
 *  ║                                                   ║    
 *  ║                                                   ║    
 *  ║     © 2023 HaoSNS™ All Rights Reserved            ║
 *  ║     官方网站: https://www.haosns.com                *
 *  ║     本代码由赣州乐易网络科技有限公司®提供             *
 *  ║                                                    *
 *  ║   未经授权禁止复制、传播或用于其他商业目的            *
 *  ║                                                   ║
 *  ╚═══════════════════════════════════════════════════╝
 */


namespace app\api\logic;

use app\common\enum\PayEnum;
use app\common\logic\BaseLogic;
use app\common\model\Config;
use app\common\model\RechargeOrder;
use app\common\model\RechargeTemplate;
use app\common\service\ConfigService;
use think\response\Json;

/**
 * 充值逻辑层
 * Class RechargeLogic
 * @package app\api\logic
 */
class RechargeLogic extends BaseLogic
{
    public static function recharge($params)
    {
        try {
            // 校验数据
            self::validateData($params);
            if(isset($params['template_id'])) {
                // 选择模板充值
                return self::rechargeByTemplate($params);
            }else{
                // 输入金额充值
                return self::rechargeByMoney($params);
            }
        } catch(\Exception $e) {
            self::setError($e->getMessage());
            return false;
        }
    }

    /**
     * @notes 校验数据
     * @param $params
     * @throws \think\Exception
     */
    public static function validateData($params)
    {
        $open = ConfigService::get('recharge', 'open');
        if(!$open) {
            throw new \think\Exception('充值功能已关闭');
        }
        if(!isset($params['pay_way'])) {
            throw new \think\Exception('请选择支付方式');
        }
        if(!isset($params['template_id']) && !isset($params['money'])) {
            throw new \think\Exception('请输入充值金额');
        }
    }

    /**
     * @notes 输入金额充值
     * @param $params
     */
    public static function rechargeByMoney($params)
    {
        $params['template_id'] = 0;
        $params['award'] = [];
        return self::addRechargeOrder($params);
    }

    /**
     * @notes 选择模板充值
     * @param $params
     * @throws \think\Exception
     */
    public static function rechargeByTemplate($params)
    {
        $template = RechargeTemplate::findOrEmpty($params['template_id']);
        if($template->isEmpty()) {
            throw new \think\Exception('充值模板不存在');
        }
        $params['money'] = $template->money;
        $params['template_id'] = $template->id;
        $params['award'] = $template->award;
        return self::addRechargeOrder($params);
    }

    /**
     * @notes 添加充值订单
     * @param $params
     */
    public static function addRechargeOrder($params)
    {
        $minAmount = ConfigService::get('recharge', 'min_amount');
        if($params['money'] < $minAmount) {
            throw new \think\Exception('最低充值金额:' . $minAmount . "元");
        }
        $award = empty($params['award']) ? '' : json_encode($params['award'], JSON_UNESCAPED_UNICODE);
        $data = [
            'sn'      => generate_sn((new RechargeOrder()),'sn'),
            'terminal'      => $params['terminal'],
            'user_id'       => $params['user_id'],
            'pay_status'    => PayEnum::UNPAID,
            'pay_way'       => $params['pay_way'],
            'order_amount'  => $params['money'],
            'template_id'   => $params['template_id'],
            'award'         => $award
        ];

        $order = RechargeOrder::create($data);
        return [
            'order_id' => $order->id,
            'from' => 'recharge'
        ];
    }
}