<?php
/**
 *  ╔═══════════════════════════════════════════════════╗
 *  ║                                                   ║
 *  ║     ██╗  ██╗   █████╗    ██████╗                  ║
 *  ║     ██║  ██║  ██╔══██╗  ██╔═══██╗                 ║
 *  ║     ███████║  ███████║  ██║   ██║                 ║
 *  ║     ██╔══██║  ██╔══██║  ██║   ██║                 ║
 *  ║     ██║  ██║  ██║  ██║  ╚██████╔╝   SNS           ║
 *  ║                                                   ║    
 *  ║                                                   ║    
 *  ║     © 2023 HaoSNS™ All Rights Reserved            ║
 *  ║     官方网站: https://www.haosns.com                *
 *  ║     本代码由赣州乐易网络科技有限公司®提供             *
 *  ║                                                    *
 *  ║   未经授权禁止复制、传播或用于其他商业目的            *
 *  ║                                                   ║
 *  ╚═══════════════════════════════════════════════════╝
 */


namespace app\api\lists;

use app\common\enum\AccountLogEnum;
use app\common\model\AccountLog;

/**
 * 账户流水列表
 * Class AccountLogLists
 * @package app\api\lists
 */
class AccountLogLists extends BaseShopDataLists
{
    /**
     * @notes 设置搜索条件
     */
    public function setWhere()
    {
        // 指定用户
        $this->searchWhere[] = ['user_id', '=', $this->userId];
        // 可提现金额类型
        if(isset($this->params['type']) && $this->params['type'] == 'bw') {
            $this->searchWhere[] = ['change_type', 'in', AccountLogEnum::getBwChangeType()];
        }
        
        // 不可提现金额类型-支出
        if(isset($this->params['type']) && $this->params['type'] == 'bnw_dec') {
            $this->searchWhere[] = ['change_type', 'in', AccountLogEnum::BNW_DEC];
        }
        // 不可提现金额类型-收入
        if(isset($this->params['type']) && $this->params['type'] == 'bnw_inc') {
            $this->searchWhere[] = ['change_type', 'in', AccountLogEnum::BNW_INC];
        }
        // 不可提现金额类型-支出 + 收入
        if(isset($this->params['type']) && $this->params['type'] == 'bnw') {
            $this->searchWhere[] = ['change_type', 'in', AccountLogEnum::getBnwChangeType()];
        }
    }

    /**
     * @notes 账户流水列表
     * @return array
     * @throws \think\db\exception\DataNotFoundException
     * @throws \think\db\exception\DbException
     * @throws \think\db\exception\ModelNotFoundException
     */
    public function lists(): array
    {
        // 设置搜索条件
        $this->setWhere();

        $field = 'id,change_type,change_amount,action,create_time,remark';
        $lists = AccountLog::field($field)
            ->where($this->searchWhere)
            ->order('id', 'desc')
            ->limit($this->limitOffset, $this->limitLength)
            ->order('id', 'desc')
            ->select()
            ->toArray();

        foreach($lists as &$item) {
            $item['type_desc'] = AccountLogEnum::getChangeTypeDesc($item['change_type']);
            $item['change_amount_desc'] = $item['action'] == AccountLogEnum::DEC ? '-' . $item['change_amount'] : '+' . $item['change_amount'];
        }

        return $lists;
    }

    /**
     * @notes 账户流水记录数
     * @return int
     */
    public function count(): int
    {
        // 设置搜索条件
        $this->setWhere();

        $count = AccountLog::where($this->searchWhere)->count();
        return $count;
    }
}