<?php
/**
 *  ╔═══════════════════════════════════════════════════╗
 *  ║                                                   ║
 *  ║     ██╗  ██╗   █████╗    ██████╗                  ║
 *  ║     ██║  ██║  ██╔══██╗  ██╔═══██╗                 ║
 *  ║     ███████║  ███████║  ██║   ██║                 ║
 *  ║     ██╔══██║  ██╔══██║  ██║   ██║                 ║
 *  ║     ██║  ██║  ██║  ██║  ╚██████╔╝   SNS           ║
 *  ║                                                   ║    
 *  ║                                                   ║    
 *  ║     © 2023 HaoSNS™ All Rights Reserved            ║
 *  ║     官方网站: https://www.haosns.com                *
 *  ║     本代码由赣州乐易网络科技有限公司®提供             *
 *  ║                                                    *
 *  ║   未经授权禁止复制、传播或用于其他商业目的            *
 *  ║                                                   ║
 *  ╚═══════════════════════════════════════════════════╝
 */
namespace app\api\validate\twoshop;


use app\common\model\GoodsSupplierCategory;
use app\common\validate\BaseValidate;

class GoodsSupplierCategoryValidate extends BaseValidate
{
    protected $rule = [
        'id' => 'require|checkId|checkDel',
        'name' => 'require|checkName',
        'sort' => 'number|max:5'
    ];

    protected $message = [
        'id.require' => 'id不能为空',
        'name.require' => '供应商分类名称不能为空',
        'sort.number' => '排序必须为数字',
        'sort.max' => '排序最大不能超过五位数',
    ];

    public function sceneAdd()
    {
        return $this->only(['name','sort']);
    }

    public function sceneDel()
    {
        return $this->only(['id']);
    }

    public function sceneEdit()
    {
        return $this->only(['id','name','sort'])
            ->append('sort','require')
            ->remove('id','checkDel');
    }

    public function sceneDetail()
    {
        return $this->only(['id'])
            ->remove('id','checkDel');
    }

    /**
     * @notes 检查供应商分类名称是否已存在
     * @param $value
     * @param $rule
     * @param $data
     * @return bool|string
     * @throws \think\db\exception\DataNotFoundException
     * @throws \think\db\exception\DbException
     * @throws \think\db\exception\ModelNotFoundException
     */
    public function checkName($value,$rule,$data)
    {
        $where[] = ['name', '=', $value];
        // 编辑的情况，要排除自身ID
        if (isset($data['id'])) {
            $where[] = ['id', '<>', $data['id']];
        }
        $result = GoodsSupplierCategory::where($where)->select()->toArray();
        if ($result) {
            return '该供应商分类已存在';
        }
        return true;
    }

    /**
     * @notes 检查供应商分类ID是否存在
     * @param $value
     * @param $rule
     * @param $data
     * @return bool|string
     */
    public function checkId($value,$rule,$data)
    {
        $result = GoodsSupplierCategory::findOrEmpty($value);
        if ($result->isEmpty()) {
            return '供应商分类不存在';
        }
        return true;
    }

    /**
     * @notes 检查供应商分类能否删除
     * @param $value
     * @param $rule
     * @param $data
     * @return bool|string
     * @throws \think\db\exception\DataNotFoundException
     * @throws \think\db\exception\DbException
     * @throws \think\db\exception\ModelNotFoundException
     * @throws \think\exception\DbException
     */
    public function checkDel($value,$rule,$data)
    {
        $result = GoodsSupplierCategory::hasWhere('goodsSupplier',['supplier_category_id'=>$value])->select()->toArray();
        if (!empty($result)) {
            return '该供应商分类正在使用中，无法删除';
        }
        return true;
    }
}