<?php
/**
 *  ╔═══════════════════════════════════════════════════╗
 *  ║                                                   ║
 *  ║     ██╗  ██╗   █████╗    ██████╗                  ║
 *  ║     ██║  ██║  ██╔══██╗  ██╔═══██╗                 ║
 *  ║     ███████║  ███████║  ██║   ██║                 ║
 *  ║     ██╔══██║  ██╔══██║  ██║   ██║                 ║
 *  ║     ██║  ██║  ██║  ██║  ╚██████╔╝   SNS           ║
 *  ║                                                   ║    
 *  ║                                                   ║    
 *  ║     © 2023 HaoSNS™ All Rights Reserved            ║
 *  ║     官方网站: https://www.haosns.com                *
 *  ║     本代码由赣州乐易网络科技有限公司®提供             *
 *  ║                                                    *
 *  ║   未经授权禁止复制、传播或用于其他商业目的            *
 *  ║                                                   ║
 *  ╚═══════════════════════════════════════════════════╝
 */



namespace app\api\validate;

use app\common\model\Cart;
use app\common\model\Goods;
use app\common\validate\BaseValidate;

/**
 * 购物车验证
 * Class CartValidate
 * @package  app\api\validate
 */
class CartValidate extends BaseValidate
{
    protected $rule = [
        'cart_id'   => 'require|checkCart',
        'goods_num' => 'require|integer|gt:0',
        'item_id'   => 'require|checkGoods',
        'selected'  => 'require|in:0,1',
    ];

    protected $message = [
        'item_id'           => '请选择商品',
        'goods_num.require' => '商品数量不能为0',
        'goods_num.gt'      => '商品数量需大于0',
        'goods_num.integer' => '商品数量需为整数',
        'cart_id.require'   => '参数错误',
        'selected.require'  => '参数错误',
        'selected.in'       => '参数错误',
    ];

    protected $scene = [
        'del'       =>  ['cart_id'],
        'add'       =>  ['item_id', 'goods_num'],
        'selected'  =>  ['cart_id', 'selected'],
        'change'    =>  ['cart_id', 'goods_num'],
    ];


    /**
     * @notes 验证购物车是否存在
     * @param $value
     * @param $rule
     * @param $data
     * @return bool|string
     */
    protected function checkCart($value, $rule, $data)
    {
        $cart = (new Cart())->getCartById($value, $data['user_id']);

        if ($cart->isEmpty()) {
            return '购物车不存在';
        }
        return true;
    }

    /**
     * @notes 验证商品
     * @param $value
     * @param $rule
     * @param $data
     * @return bool|string
     */
    protected function checkGoods($value, $rule, $data)
    {
        $goods = (new Goods())->alias('g')
            ->field(['g.status', 'g.delete_time', 'gi.stock'])
            ->join('goods_item gi', 'gi.goods_id = g.id')
            ->where(['gi.id' => $value])
            ->find();

        if (empty($goods) || $goods['status'] == 0 || $goods['delete_time'] > 0) {
            return '商品已下架';
        }

        $cart = (new Cart())->getCartByItem($data['item_id'], $data['user_id']);

        $cartNum = $data['goods_num'] + $cart['goods_num'] ?? 0;
        if ($goods['stock'] < $cartNum) {
            return '很抱歉,商品库存不足';
        }
        return true;
    }

}