<?php
/**
 *  ╔═══════════════════════════════════════════════════╗
 *  ║                                                   ║
 *  ║     ██╗  ██╗   █████╗    ██████╗                  ║
 *  ║     ██║  ██║  ██╔══██╗  ██╔═══██╗                 ║
 *  ║     ███████║  ███████║  ██║   ██║                 ║
 *  ║     ██╔══██║  ██╔══██║  ██║   ██║                 ║
 *  ║     ██║  ██║  ██║  ██║  ╚██████╔╝   SNS           ║
 *  ║                                                   ║    
 *  ║                                                   ║    
 *  ║     © 2023 HaoSNS™ All Rights Reserved            ║
 *  ║     官方网站: https://www.haosns.com                *
 *  ║     本代码由赣州乐易网络科技有限公司®提供             *
 *  ║                                                    *
 *  ║   未经授权禁止复制、传播或用于其他商业目的            *
 *  ║                                                   ║
 *  ╚═══════════════════════════════════════════════════╝
 */

namespace app\common\logic;

use app\adminapi\logic\user\UserLogic as UserUserLogic;
use app\common\{enum\PayEnum, model\Order, model\User, model\UserLevel};
use app\common\service\ConfigService;

/**
 * 用户逻辑类
 * Class UserLogic
 * @package app\common\logic
 */
class UserLogic extends BaseLogic
{
    /**
     * @notes 注册奖励
     * @param int $userId
     */
    public static function registerAward(int $userId)
    {
        //默认等级
        self::defaultUserLevel($userId);

        // 邀请奖励
        self::inviteAward($userId);
    }

    /**
     * @notes 注册后调判断是否有邀请人，有邀请人给予奖励
     * @param $userId
  
     */
    public static function inviteAward($userId)
    {
        
        $invite_friend_gift_money = ConfigService::get('config', 'invite_friend_gift_money', 0);
        
        $invite_gift_money = ConfigService::get('config', 'invite_gift_money', 0);

        $user = User::find($userId);
        $userUserLogic = new UserUserLogic();
        if($user && $user['inviter_id']  &&  $invite_gift_money){
            $userUserLogic->adjustUserWallet([
                'user_id' => $user['inviter_id'],
                'type'    => 1,
                'action'  => 1,
                'num'     => $invite_gift_money,
                'remark'  => '邀请好友奖励'
            ]);
        }
        if($invite_friend_gift_money){
            $userUserLogic->adjustUserWallet([
                'user_id' => $userId,
                'type'    => 1,
                'action'  => 1,
                'num'     => $invite_friend_gift_money,
                'remark'  => '邀请奖励'
            ]);
        }
        

    }

    /**
     * @notes 注册后调整默认等级
     * @param $userId
     * @throws \think\db\exception\DataNotFoundException
     * @throws \think\db\exception\DbException
     * @throws \think\db\exception\ModelNotFoundException
     */
    public static function defaultUserLevel($userId)
    {
        $level = UserLevel::where(['rank' => 1])->find();
        if ($level) {
            User::where(['id' => $userId])->update(['level' => $level->id]);
        }

    }


    /**
     * @notes 格式化会员等级
     * @param $condition
     */
    public static function formatLevelCondition($condition)
    {
        if(empty($condition)){
            return [];
        }
        $condition['condition_type'] = (int)$condition['condition_type'];
        if(!isset($condition['is_single_money'])){
            $condition['is_single_money'] = $condition['single_money'] > 0 ? 1 :0;
        }
        if(!isset($condition['is_total_money'])){
            $condition['is_total_money'] = $condition['total_money'] > 0 ? 1 :0;
        }
        if(!isset($condition['is_total_num'])){
            $condition['is_total_num'] = $condition['total_num'] > 0 ? 1 :0;
        }
        return $condition;

    }

    /**
     * @notes 更新会员等级 todo 该方法调在更新用户的累计金额和累计订单数后调用
     * @param int $userId
     * @return bool
     */
    public static function updateLevel(int $userId)
    {

        $user = User::with('user_level')->find($userId);
        // 获取用户当前等级的rank
        $currentRank = $user->user_level ? $user->user_level->rank : 0;
        $levelList = UserLevel::where('rank', '>', $currentRank)
            ->order('rank asc')
            ->select();
        //没有比会员当前高的等级，直接断掉
        if (empty($levelList)) {
            return true;
        }
        $orderAmount = Order::where(['user_id' => $userId, 'pay_status' => PayEnum::ISPAID])
            ->order('order_amount desc')
            ->value('order_amount');

        //从最低等级开始遍历
        foreach ($levelList as $level) {
            $condition = self::formatLevelCondition($level['condition']);

            // 如果condition为空，跳过该等级
            if (empty($condition)) {
                continue;
            }

            $conditionType = $condition['condition_type'] ?? 0;

            $singleMoney = $condition['single_money'] ?? 0;
            $isSingleMoney = $condition['is_single_money'] ?? 0;
            $totalMoney = $condition['total_money'] ?? 0;
            $isTotalMoney = $condition['is_total_money'] ?? 0;
            $totalNum = $condition['total_num'] ?? 0;
            $isTotalNum = $condition['is_total_num'] ?? 0;
            //数据异常不处理
            if( 0 == $isSingleMoney && 0 == $isTotalMoney  && 0 == $isTotalNum){
                continue;
            }

            //满足其中任意条件
            if(0 == $conditionType){

                $singleMoneyBoole = false;      //满足单笔消费条件
                $totalMoneyBoole = false;       //满足累计消费金额条件
                $totalNumBoole = false;         //累计消费次数条件

                //是否满足单笔消费条件
                if (1 == $isSingleMoney && $singleMoney > 0 && $orderAmount >= $singleMoney) {
                    $singleMoneyBoole = true;
                }
                //是否满足累计消费金额
                if (1 == $isTotalMoney && $totalMoney > 0 && $user->total_order_amount >= $totalMoney) {
                    $totalMoneyBoole = true;
                }
                //是否满足消费次数
                if (1 == $isTotalNum && $totalNum > 0 && $user->total_order_num >= $totalNum) {
                    $totalNumBoole = true;
                }

                //满足其中任意条件，升级到该等级并退出循环（避免重复升级）
                if ($singleMoneyBoole || $totalMoneyBoole || $totalNumBoole) {
                    $user->level = $level->id;
                    $user->save();
                    break;
                }

            }else{
                //
                $singleMoneyBoole = true;      //满足单笔消费条件
                $totalMoneyBoole = true;       //满足累计消费金额条件
                $totalNumBoole = true;         //累计消费次数条件

                //判断不满单笔消费条件
                if (1 == $isSingleMoney && $singleMoney > 0 && $orderAmount < $singleMoney) {
                    $singleMoneyBoole = false;
                }
                //判断不满累计消费金额条件
                if (1 == $isTotalMoney && $totalMoney > 0 && $user->total_order_amount < $totalMoney) {
                    $totalMoneyBoole = false;
                }
                //判断不满足消费次数条件
                if (1 == $isTotalNum && $totalNum > 0 && $user->total_order_num < $totalNum) {
                    $totalNumBoole = false;
                }

                //满足勾选的全部条件，升级到该等级并退出循环（避免重复升级）
                if ($singleMoneyBoole && $totalMoneyBoole && $totalNumBoole) {
                    $user->level = $level->id;
                    $user->save();
                    break;
                }

            }

        }
        return true;
    }

    /**
     * @notes 根据用户经验值更新用户等级
     * @param int $userId
     * @return bool
     */
    public static function updateLevelByExperience(int $userId)
    {
        $user = User::find($userId);
        if (!$user) {
            return false;
        }

        // 1. 获取用户当前的物理积分值，确保计算基数正确
        $userIntegral = (int)($user->user_integral ?? 0);

        // 2. 获取所有等级，必须按 rank (等级权重) 从小到大排序
        // 这样我们才能从 Lv.1 开始依次往上检查
        $levelList = UserLevel::order('rank asc')->select();

        if ($levelList->isEmpty()) {
            return true;
        }

        // 3. 寻找匹配等级
        $targetLevel = $levelList[0]; // 默认至少是第一个等级 (Lv.1)
        
        foreach ($levelList as $level) {
            $threshold = (int)($level->experience ?? 0);
            if ($userIntegral >= $threshold) {
                // 如果积分达到了这个等级的门槛，暂时记录它，继续往后看有没有更高的
                $targetLevel = $level;
            } else {
                // 一旦积分不足以支撑当前循环到的等级，说明上一个记录的 targetLevel 就是最高能达到的等级
                break;
            }
        }

        // 4. 执行更新
        if ((int)$targetLevel->id !== (int)$user->level) {
            $user->level = $targetLevel->id;
            $user->save();
        }

        return true;
    }

}