<?php
/**
 *  ╔═══════════════════════════════════════════════════╗
 *  ║                                                   ║
 *  ║     ██╗  ██╗   █████╗    ██████╗                  ║
 *  ║     ██║  ██║  ██╔══██╗  ██╔═══██╗                 ║
 *  ║     ███████║  ███████║  ██║   ██║                 ║
 *  ║     ██╔══██║  ██╔══██║  ██║   ██║                 ║
 *  ║     ██║  ██║  ██║  ██║  ╚██████╔╝   SNS           ║
 *  ║                                                   ║    
 *  ║                                                   ║    
 *  ║     © 2023 HaoSNS™ All Rights Reserved            ║
 *  ║     官方网站: https://www.haosns.com                *
 *  ║     本代码由赣州乐易网络科技有限公司®提供             *
 *  ║                                                    *
 *  ║   未经授权禁止复制、传播或用于其他商业目的            *
 *  ║                                                   ║
 *  ╚═══════════════════════════════════════════════════╝
 */



namespace app\api\validate\twoshop;


use app\common\enum\DeliveryEnum;
use app\common\enum\OrderEnum;
use app\common\model\User;
use app\common\validate\BaseValidate;

/**
 * 订单提交验证
 * Class OrderValidate
 * @package  app\api\validate
 */
class PlaceOrderValidate extends BaseValidate
{

    protected $rule = [
        // 下单用户
        'user_id' => 'checkUser',
        // 下单来源(立即购买/购物车购买)
        'source'    => 'require',
        // 下单动作(结算/下单)
        'action'    => 'require',
        // 订单类型
        'order_type' => 'require|checkGoods|checkOrderType',
        // 配送方式
        'delivery_type' => 'require|checkDeliveryType'
    ];


    protected $message = [
        'source.require'    => '下单来源缺失',
        'action.require'    => '下单动作缺失',
        'order_type.require'     => '订单类型缺失',
        'delivery_type.require'     => '配送参数缺失',
    ];

    /**
     * @notes 检测用户是否登录
     * @param $value
     * @param $rule
     * @param $data
     * @return bool|string
     */
    public static function checkUser($value, $rule, $data)
    {
        $user = User::findOrEmpty($value);
        if ($user->isEmpty()) {
            return '请先登录';
        }
        return true;
    }

    /**
     * @notes 检查配送方式
     * @param $value
     * @param $rule
     * @param $data
     */
    public function checkDeliveryType($value, $rule, $data)
    {
        if (!in_array($data['delivery_type'],DeliveryEnum::DELIVERY_TYPE)) {
            return '无效的配送方式';
        }
        return true;
    }

    /**
     * @notes 验证商品参数
     * @param $value
     * @param $rule
     * @param $data
     * @return bool|string
     */
    public function checkGoods($value, $rule, $data)
    {
        // dd($data);

        if (!isset($data['goods'])) {
            return '商品信息参数缺失';
        }

        if (!is_array($data['goods'])) {
            return '商品信息格式不正确';
        }

        if (!isset($data['goods']['item_id']) || !isset($data['goods']['goods_num'])) {
            return '商品信息参数不完整';
        }
        
        return true;
    }

    /**
     * @notes 校验订单类型
     * @param $value
     * @param $rule
     * @param $data
     * @return bool|string
     */
    public function checkOrderType($value, $rule, $data)
    {
        if (!in_array($value, OrderEnum::ORDER_TYPE)) {
            return '无效的订单类型';
        }
        return true;
    }
}