<?php
/**
 *  ╔═══════════════════════════════════════════════════╗
 *  ║                                                   ║
 *  ║     ██╗  ██╗   █████╗    ██████╗                  ║
 *  ║     ██║  ██║  ██╔══██╗  ██╔═══██╗                 ║
 *  ║     ███████║  ███████║  ██║   ██║                 ║
 *  ║     ██╔══██║  ██╔══██║  ██║   ██║                 ║
 *  ║     ██║  ██║  ██║  ██║  ╚██████╔╝   SNS           ║
 *  ║                                                   ║    
 *  ║                                                   ║    
 *  ║     © 2023 HaoSNS™ All Rights Reserved            ║
 *  ║     官方网站: https://www.haosns.com                *
 *  ║     本代码由赣州乐易网络科技有限公司®提供             *
 *  ║                                                    *
 *  ║   未经授权禁止复制、传播或用于其他商业目的            *
 *  ║                                                   ║
 *  ╚═══════════════════════════════════════════════════╝
 */
namespace app\api\validate\twoshop;


use app\common\model\GoodsBrand;
use app\common\validate\BaseValidate;

class GoodsBrandValidate extends BaseValidate
{
    protected $rule = [
        'id' => 'require|checkId|checkDel',
        'name' => 'require|checkName|max:8',
        'is_show' => 'in:0,1',
        'sort' => 'number',
    ];

    protected $message = [
        'id.require' => '品牌id不能为空',
        'name.require' => '品牌名称不能为空',
    ];

    public function sceneAdd()
    {
        return $this->only(['name','is_show','sort']);
    }

    public function sceneDel()
    {
        return $this->only(['id']);
    }

    public function sceneEdit()
    {
        return $this->only(['id','name','is_show','sort'])
            ->remove('id','checkDel');
    }

    public function sceneStatus()
    {
        return $this->only(['id','is_show'])
            ->remove('id','checkDel');
    }

    public function sceneDetail()
    {
        return $this->only(['id'])
            ->remove('id','checkDel');
    }

    /**
     * @notes 验证品牌是否已存在
     * @param $value
     * @param $rule
     * @param $data
     * @return bool|string
     * @throws \think\db\exception\DataNotFoundException
     * @throws \think\db\exception\DbException
     * @throws \think\db\exception\ModelNotFoundException
     */
    public function checkName($value,$rule,$data)
    {
        $where[] = ['name', '=', $value];
        // 编辑的情况，要排除自身ID
        if (isset($data['id'])) {
            $where[] = ['id', '<>', $data['id']];
        }
        $result = GoodsBrand::where($where)->select()->toArray();
        if ($result) {
            return '该品牌已存在';
        }
        return true;
    }

    /**
     * @notes 验证品牌是否存在
     * @param $value
     * @param $rule
     * @param $data
     * @return bool|string
     */
    public function checkId($value,$rule,$data)
    {
        $result = GoodsBrand::findOrEmpty($value);
        if ($result->isEmpty()) {
            return '品牌不存在';
        }
        return true;
    }

    /**
     * @notes 验证品牌能否删除
     * @param $value
     * @param $rule
     * @param $data
     * @return bool|string
     * @throws \think\db\exception\DataNotFoundException
     * @throws \think\db\exception\DbException
     * @throws \think\db\exception\ModelNotFoundException
     * @throws \think\exception\DbException
     */
    public function checkDel($value,$rule,$data)
    {
        $result = GoodsBrand::hasWhere('goods',['brand_id'=>$value])->select()->toArray();
        if (!empty($result)) {
            return '该品牌正在使用中，无法删除';
        }
        return true;
    }
}