<?php
/**
 *  ╔═══════════════════════════════════════════════════╗
 *  ║                                                   ║
 *  ║     ██╗  ██╗   █████╗    ██████╗                  ║
 *  ║     ██║  ██║  ██╔══██╗  ██╔═══██╗                 ║
 *  ║     ███████║  ███████║  ██║   ██║                 ║
 *  ║     ██╔══██║  ██╔══██║  ██║   ██║                 ║
 *  ║     ██║  ██║  ██║  ██║  ╚██████╔╝   SNS           ║
 *  ║                                                   ║    
 *  ║                                                   ║    
 *  ║     © 2023 HaoSNS™ All Rights Reserved            ║
 *  ║     官方网站: https://www.haosns.com                *
 *  ║     本代码由赣州乐易网络科技有限公司®提供             *
 *  ║                                                    *
 *  ║   未经授权禁止复制、传播或用于其他商业目的            *
 *  ║                                                   ║
 *  ╚═══════════════════════════════════════════════════╝
 */


namespace app\common\logic;

use app\common\enum\AccountLogEnum;
use app\common\model\AccountLog;
use app\common\model\User;

/**
 * 账户流水记录逻辑层
 * Class AccountLogLogic
 * @package app\common\logic
 */
class AccountLogLogic extends BaseLogic
{
    /**
     * @notes 添加账户流水记录
     * @param $userId //会员ID
     * @param $changeType //变动类型(需在AccountLogEnum预先定义好)
     * @param $action //变动动作(AccountLogEnum中有定义好常量 1-减少 2-增加)
     * @param $changeAmount //变动数量
     * @param $associationSn //关联单号(例如：订单号、充值单号、提现单号等)
     * @param string $remark 备注
     * @param array $feature 预留字段，方便存更多其它信息
     * @return false
     */
    public static function add($userId, $changeType, $action, $changeAmount, $associationSn = '', $remark = '', $feature = [])
    {
        $user = User::findOrEmpty($userId);
        if($user->isEmpty()) {
            return false;
        }

        // dd(AccountLogEnum::BNW);
        // dd(AccountLogEnum::BW);
        $changeObject = AccountLogEnum::getChangeObject($changeType);
        if(!$changeObject) {
            return false;
        }
        switch ($changeObject) {
            // 可提现余额(用户收入)
            case AccountLogEnum::BW:
                $left_amount = $user->user_earnings;
                break;
            // 不可提现余额
            case AccountLogEnum::BNW:
                $left_amount = $user->user_money;
                break;
            // 成长值
            case AccountLogEnum::GROWTH:
                $left_amount = $user->user_growth;
                break;
        }

        $accountLog = new AccountLog();
        $data = [
            'sn' => generate_sn($accountLog, 'sn', 20),
            'user_id' => $userId,
            'change_object' => $changeObject,
            'change_type' => $changeType,
            'action' => $action,
            'left_amount' => $left_amount,
            'change_amount' => $changeAmount,
            'association_sn' => $associationSn,
            'remark' => $remark,
            'feature' => $feature ? json_encode($feature, JSON_UNESCAPED_UNICODE) : '',
        ];
        return $accountLog->save($data);
    }
}