<?php
/**
 *  ╔═══════════════════════════════════════════════════╗
 *  ║                                                   ║
 *  ║     ██╗  ██╗   █████╗    ██████╗                  ║
 *  ║     ██║  ██║  ██╔══██╗  ██╔═══██╗                 ║
 *  ║     ███████║  ███████║  ██║   ██║                 ║
 *  ║     ██╔══██║  ██╔══██║  ██║   ██║                 ║
 *  ║     ██║  ██║  ██║  ██║  ╚██████╔╝   SNS           ║
 *  ║                                                   ║    
 *  ║                                                   ║    
 *  ║     © 2023 HaoSNS™ All Rights Reserved            ║
 *  ║     官方网站: https://www.haosns.com                *
 *  ║     本代码由赣州乐易网络科技有限公司®提供             *
 *  ║                                                    *
 *  ║   未经授权禁止复制、传播或用于其他商业目的            *
 *  ║                                                   ║
 *  ╚═══════════════════════════════════════════════════╝
 */


declare(strict_types=1);

namespace app\common\service;

use app\common\model\Config;
use think\facade\Cache;

class ConfigService
{

    /**
     * @notes
     * @param string $type
     * @param string $name
     * @return string
     * @author lgp
     * @date 2022/10/19 16:13
     */
    public static function getCacheKey($type, $name): string
    {
        return 'config' . '-' . $type . '-' . $name;
    }

    /**
     * @notes 设置配置值
     * @param string $type
     * @param string $name
     * @param mixed $value
     * @return mixed
     */
    public static function set($type, $name, $value)
    {
        //刷新缓存
        $cacheKey = self::getCacheKey($type, $name);
        Cache::delete($cacheKey);

        $original = $value;
        // 数组数据进行json编码
        if (is_array($value)) {
            // JSON_UNESCAPED_UNICODE 不对中文进行unicode编码
            $value = json_encode($value, JSON_UNESCAPED_UNICODE);
        }
        $data = Config::where(['type' => $type, 'name' => $name])->findOrEmpty();
        if ($data->isEmpty()) {
            // 没有则添加
            Config::create([
                'type' => $type,
                'name' => $name,
                'value' => $value,
            ]);
        } else {
            // 有则修改
            $data->value = $value;
            $data->save();
        }
        // 返回原始值
        return $original;
    }

    /**
     * @notes
     * @param $type
     * @param string $name
     * @param null $defaultValue
     * @return mixed
     * @author lgp
     * @date 2022/10/19 16:16
     */
    public static function get($type, $name = '', $defaultValue = null)
    {
        //获取缓存
        $CacheKey = self::getCacheKey($type, $name);
        // $result = Cache::get($CacheKey);
        // $value = $result['config_server'] ?? null;
        // if ($value !== null) {
        //     return $value;
        // }

        //单项配置
        if ($name) {
            $result = Config::where(['type' => $type, 'name' => $name])->value('value');
            // dd($result);
            //json类型解析
            if (!is_null($result)) {
                $json = json_decode($result, true);
                $result = json_last_error() === JSON_ERROR_NONE ? $json : $result;
            }
            //获取默认配置
            if ($result === null) {
                $result = $defaultValue;
            }
            //获取本地配置文件配置
            if ($result === null) {
                $result = config('project.' . $type . '.' . $name);
            }
            Cache::set($CacheKey, ['config_server' => $result]);
            return $result;
        }

        //多项配置
        $result = Config::where(['type' => $type])->column('value', 'name');
        if (is_array($result)) {
            foreach ($result as $k => $v) {
                $json = json_decode($v, true);
                if (json_last_error() === JSON_ERROR_NONE) {
                    $result[$k] = $json;
                }
            }
        }
        //读取默认配置
        if ($result === null) {
            $result = $defaultValue;
        }
        Cache::set($CacheKey, ['config_server' => $result]);
        return $result;
    }

}