<?php
/**
 *  ╔═══════════════════════════════════════════════════╗
 *  ║                                                   ║
 *  ║     ██╗  ██╗   █████╗    ██████╗                  ║
 *  ║     ██║  ██║  ██╔══██╗  ██╔═══██╗                 ║
 *  ║     ███████║  ███████║  ██║   ██║                 ║
 *  ║     ██╔══██║  ██╔══██║  ██║   ██║                 ║
 *  ║     ██║  ██║  ██║  ██║  ╚██████╔╝   SNS           ║
 *  ║                                                   ║    
 *  ║                                                   ║    
 *  ║     © 2023 HaoSNS™ All Rights Reserved            ║
 *  ║     官方网站: https://www.haosns.com                *
 *  ║     本代码由赣州乐易网络科技有限公司®提供             *
 *  ║                                                    *
 *  ║   未经授权禁止复制、传播或用于其他商业目的            *
 *  ║                                                   ║
 *  ╚═══════════════════════════════════════════════════╝
 */


namespace app\common\enum;

/**
 * 会员账户流水变动表枚举
 * Class AccountLogEnum
 * @package app\common\enum
 */
class AccountLogEnum
{
    /**
     * 变动类型命名规则：对象_动作_简洁描述
     * 动作 DEC-减少 INC-增加
     * 对象 BNW-不可提现余额(充值等) BW-可提现余额 GROWTH-成长值 
     * BNW - BALANCE_NOT_WITHDRAWABLE 不可提现余额
     * BW - BALANCE_WITHDRAWABLE 可提现余额
     * 可提现余额类型 100-199(减少) 200-299(增加)
     * 不可提现余额类型 300-399(减少) 400-499(增加)
     * 成长值类型 500-599(减少) 600-699(增加)
     * 例：
     *  const BNW_DEC_ADMIN = 100; // 管理员减少余额
     *  const BNW_INC_ADMIN = 200; // 管理员增加余额
     */

    /**
     * 变动对象
     * BW 可提现余额
     * BNW 不可提现余额
     * GROWTH 成长值
     */
    const BW = 1;
    const BNW = 2;
    const GROWTH = 3;


    /**
     * 动作
     * DEC 减少
     * INC 增加
     */
    const DEC = 1;
    const INC = 2;

    /**
     * 不可提现余额减少类型
     */
    const BNW_DEC_ADMIN = 100;
    const BNW_DEC_ORDER = 101;
    const BNW_DEC_TRANSFER = 102;

    /**
     * 不可提现余额增加类型
     */
    const BNW_INC_ADMIN = 200;
    const BNW_INC_RECHARGE = 201;
    const BNW_INC_CANCEL_ORDER = 202;
    const BNW_INC_AFTER_SALE = 203;
    const BNW_INC_WITHDRAW = 204;
    const BNW_INC_RECHARGE_GIVE = 205;
    const BNW_INC_TRANSFER = 206;
    const BNW_INC_LOTTERY = 207;

    /**
     * 可提现余额减少类型
     */
    const BW_DEC_WITHDRAWAL = 300;
    const BW_DEC_ADMIN      = 301;

    /**
     * 可提现余额增加类型
     */
    const BW_INC_REFUSE_WITHDRAWAL      = 400;
    const BW_INC_DISTRIBUTION_SETTLE    = 401;
    const BW_INC_PAYMENT_FAIL           = 402;
    const BW_INC_TRANSFER_FAIL          = 403;
    const BW_INC_ADMIN                  = 404;
    const BW_INC_SIGN                   = 405;
    const BW_INC_SELL                   = 406;//卖东西的收入
    const BW_INC_ARTICLE_REWARD         = 407;//文章打赏
    const BW_INC_PROMISE_MONEY_BACK     = 408;//保证金退还
    /**
     * 成长值减少类型
     */
    const GROWTH_DEC_ADMIN = 500;

    /**
     * 成长值增加类型
     */
    const GROWTH_INC_ADMIN = 600;
    const GROWTH_INC_SIGN = 601;
    const GROWTH_INC_RECHARGE = 602;
    const GROWTH_INC_ORDER = 603;


    /**
     * 不可提现余额（减少类型汇总）
     */
    const BNW_DEC = [
        self::BNW_DEC_ADMIN,
        self::BNW_DEC_ORDER,
        self::BNW_DEC_TRANSFER
    ];

    /**
     * 不可提现余额（增加类型汇总）
     */
    const BNW_INC = [
        self::BNW_INC_ADMIN,
        self::BNW_INC_RECHARGE,
        self::BNW_INC_CANCEL_ORDER,
        self::BNW_INC_AFTER_SALE,
        self::BNW_INC_WITHDRAW,
        self::BNW_INC_RECHARGE_GIVE,
        self::BNW_INC_TRANSFER,
        self::BNW_INC_LOTTERY,
        self::BW_INC_SIGN
    ];

    /**
     * 可提现余额（减少类型汇总）
     */
    const BW_DEC = [
        self::BW_DEC_ADMIN,
        self::BW_DEC_WITHDRAWAL,
    ];

    /**
     * 可提现余额(增加类型汇总）
     */
    const BW_INC = [
        self::BW_INC_REFUSE_WITHDRAWAL,
        self::BW_INC_DISTRIBUTION_SETTLE,
        self::BW_INC_PAYMENT_FAIL,
        self::BW_INC_TRANSFER_FAIL,
        self::BW_INC_ADMIN,
        self::BW_INC_SELL,
        self::BW_INC_ARTICLE_REWARD,
        self::BW_INC_PROMISE_MONEY_BACK
    ];

    /**
     * 成长值(减少类型汇总)
     */
    const GROWTH_DEC = [
        self::GROWTH_DEC_ADMIN
    ];

    /**
     * 成长值(增加类型汇总)
     */
    const GROWTH_INC = [
        self::GROWTH_INC_ADMIN,
        self::GROWTH_INC_SIGN,
        self::GROWTH_INC_RECHARGE,
        self::GROWTH_INC_ORDER,
    ];


    /**
     * @notes 动作描述
     * @param $action
     * @param false $flag
     * @return string|string[]
     */
    public static function getActionDesc($action, $flag = false)
    {
        $desc = [
            self::DEC => '减少',
            self::INC => '增加',
        ];
        if($flag) {
            return $desc;
        }
        return $desc[$action] ?? '';
    }

    /**
     * @notes 获取变动类型描述
     * @param $changeType
     * @param false $flag
     * @return string|string[]
     */
    public static function getChangeTypeDesc($changeType, $flag = false)
    {
        $desc = [
            self::BNW_DEC_ADMIN => '管理员减少金币',
            self::BNW_DEC_ORDER => '下单扣减金币',
            self::BNW_DEC_TRANSFER => '转账扣减金币',
            self::BNW_INC_ADMIN => '管理员增加金币',
            self::BNW_INC_RECHARGE => '充值增加金币',
            self::BNW_INC_CANCEL_ORDER => '取消订单退还金币',
            self::BNW_INC_AFTER_SALE => '售后退还金币',
            self::BNW_INC_WITHDRAW => '提现至金币',
            self::BNW_INC_RECHARGE_GIVE => '充值赠送金币',
            self::BNW_INC_TRANSFER => '转账增加金币',
            self::BW_DEC_ADMIN      => '管理员减少金币',
            self::BW_INC_REFUSE_WITHDRAWAL => '拒绝提现回退金币',
            self::BW_INC_PAYMENT_FAIL => '付款失败回退金币',
            self::BW_INC_TRANSFER_FAIL => '转账失败回退金币',
            self::BW_INC_ADMIN          => '管理员增加金币',
            self::BW_INC_SELL           => '商品卖出收入',
            self::GROWTH_DEC_ADMIN => '管理员减少成长值',
            self::GROWTH_INC_ADMIN => '管理员增加成长值',
            self::GROWTH_INC_SIGN => '签到赠送成长值',
            self::GROWTH_INC_RECHARGE => '充值赠送成长值',
            self::GROWTH_INC_ORDER => '下单赠送成长值',
            self::BW_INC_SIGN   => '签到送金币',
            self::BW_INC_ARTICLE_REWARD => '文章打赏',
            self::BW_INC_PROMISE_MONEY_BACK => '保证金退还'

        ];
        if($flag) {
            return $desc;
        }
        return $desc[$changeType] ?? '';
    }

    /**
     * @notes 获取不可提现余额类型描述
     * @return string|string[]
     */
    public static function getBnwChangeTypeDesc()
    {
        $bnwChangeType = self::getBnwChangeType();
        $changeTypeDesc = self::getChangeTypeDesc('',true);
        $bnwChangeTypeDesc = array_filter($changeTypeDesc, function($key)  use ($bnwChangeType) {
            return in_array($key, $bnwChangeType);
        }, ARRAY_FILTER_USE_KEY);
        return $bnwChangeTypeDesc;
    }

    /**
     * @notes 获取可提现余额类型描述
     * @return string|string[]
     * @date 2022/3/28 11:18
     */
    public static function getBwChangeTypeDesc()
    {
        $bwChangeType = self::getBwChangeType();
        $changeTypeDesc = self::getChangeTypeDesc('',true);
        $bwChangeTypeDesc = array_filter($changeTypeDesc, function($key)  use ($bwChangeType) {
            return in_array($key, $bwChangeType);
        }, ARRAY_FILTER_USE_KEY);
        return $bwChangeTypeDesc;
    }

    /**
     * @notes 获取可提现余额变动类型
     * @return int[]
     */
    public static function getBwChangeType()
    {
        $bw = array_merge(self::BW_DEC, self::BW_INC);
        return $bw;
    }

    /**
     * @notes 获取不可提现余额变动类型
     * @return int[]
     */
    public static function getBnwChangeType()
    {
        $bnw = array_merge(self::BNW_DEC, self::BNW_INC);
        return $bnw;
    }

    /**
     * @notes 获取余额变动类型
     * @return int[]
     */
    public static function getBalanceChangeType()
    {
        $bw = self::getBwChangeType();
        $bnw = self::getBnwChangeType();
        $balance = array_merge($bw, $bnw);
        return $balance;
    }

    /**
     * @notes 获取成长值变动类型
     * @return int[]
     */
    public static function getGrowthChangeType()
    {
        $growth = array_merge(self::GROWTH_DEC, self::GROWTH_INC);
        return $growth;
    }


    /**
     * @notes 获取变动对象
     * @param $changeType
     * @return false|string
     */
    public static function getChangeObject($changeType)
    {
        // 可提现余额
        $bw = self::getBwChangeType();
        if(in_array($changeType, $bw)) {
            return self::BW;
        }

        // 不可提现余额
        $bnw = self::getBnwChangeType();
        if(in_array($changeType, $bnw)) {
            return self::BNW;
        }

        // 成长值
        $growth = self::getGrowthChangeType();
        if(in_array($changeType, $growth)) {
            return self::GROWTH;
        }

        return false;
    }
}