<?php
/**
 *  ╔═══════════════════════════════════════════════════╗
 *  ║                                                   ║
 *  ║     ██╗  ██╗   █████╗    ██████╗                  ║
 *  ║     ██║  ██║  ██╔══██╗  ██╔═══██╗                 ║
 *  ║     ███████║  ███████║  ██║   ██║                 ║
 *  ║     ██╔══██║  ██╔══██║  ██║   ██║                 ║
 *  ║     ██║  ██║  ██║  ██║  ╚██████╔╝   SNS           ║
 *  ║                                                   ║    
 *  ║                                                   ║    
 *  ║     © 2023 HaoSNS™ All Rights Reserved            ║
 *  ║     官方网站: https://www.haosns.com                *
 *  ║     本代码由赣州乐易网络科技有限公司®提供             *
 *  ║                                                    *
 *  ║   未经授权禁止复制、传播或用于其他商业目的            *
 *  ║                                                   ║
 *  ╚═══════════════════════════════════════════════════╝
 */



namespace app\api\validate;


use app\common\enum\AfterSaleEnum;
use app\common\enum\OrderEnum;
use app\common\enum\YesNoEnum;
use app\common\model\AfterSale;
use app\common\model\Order;
use app\common\service\ConfigService;
use app\common\validate\BaseValidate;

/**
 * 订单验证
 * Class OrderValidate
 * @package  app\api\validate
 */
class OrderValidate extends BaseValidate
{

    protected $rule = [
        'id' => 'require|checkOrder'
    ];


    protected $message = [
        'id.require' => '参数缺失',
    ];


    public function sceneDetail()
    {
        return $this->only(['id']);
    }


    public function sceneCancel()
    {
        return $this->only(['id'])->append('id', 'checkCancel');
    }


    public function sceneConfirm()
    {
        return $this->only(['id'])->append('id', 'checkConfirm');
    }

    public function sceneOrderTraces()
    {
        return $this->only(['id'])->append('id','checkTraces');
    }

    public function sceneDel()
    {
        return $this->only(['id'])->append('id','checkDel');
    }


    //验证订单
    public function checkOrder($value, $rule, $data)
    {
        $order = (new Order())->getUserOrderById($value, $data['user_id']);

        if ($order->isEmpty()) {
            return '订单不存在';
        }

        return true;
    }


    /**
     * @notes 验证订单能否取消
     * @param $value
     * @param $rule
     * @param $data
     * @return bool|string
     */
    public function checkCancel($value, $rule, $data)
    {
        $order = (new Order())->getUserOrderById($value, $data['user_id']);

        //已发货订单不可取消
        if ($order->isEmpty() || $order['order_status'] > OrderEnum::STATUS_WAIT_DELIVERY) {
            return '很抱歉!订单无法取消';
        }

        // 用户未支付时，允许任意取消；已支付时，在允许取消的时间内订单未发货则允许取消；
        if ($order['order_status'] == OrderEnum::STATUS_WAIT_DELIVERY) {
            $ableCancelOrder = ConfigService::get('transaction', 'cancel_unshipped_orders');
            if ($ableCancelOrder == YesNoEnum::NO) {
                return '订单不可取消';
            }
            $configTime = ConfigService::get('transaction', 'cancel_unshipped_orders_times');
            $ableCancelTime = strtotime($order['create_time']) + ($configTime * 60);
            if (time() > $ableCancelTime) {
                return '订单不可取消';
            }
        }

        return true;
    }


    /**
     * @notes 验证能否确认收货
     * @param $value
     * @param $rule
     * @param $data
     * @return bool|string
     */
    public function checkConfirm($value, $rule, $data)
    {
        $order = (new Order())->getUserOrderById($value, $data['user_id']);

        if ($order->isEmpty()) {
            return '订单不存在';
        }

        if ($order['order_status'] < OrderEnum::STATUS_WAIT_RECEIVE) {
            return '订单未发货';
        }

        if ($order['order_status'] == OrderEnum::STATUS_FINISH) {
            return '订单已完成';
        }

        return true;
    }

    /**
     * @notes 检查订单是否有物流信息
     * @param $value
     * @param $rule
     * @param $data
     * @return bool|string
     */
    public function checkTraces($value, $rule, $data)
    {
        $order = (new Order())->getUserOrderById($value, $data['user_id']);

        if ($order->isEmpty()) {
            return '订单不存在';
        }

        if ($order['order_status'] < OrderEnum::STATUS_WAIT_RECEIVE) {
            return '订单未发货';
        }

        return true;
    }

    /**
     * @notes 检查订单是否可以删除
     * @param $value
     * @param $rule
     * @param $data
     * @return bool|string
     * @throws \think\db\exception\DataNotFoundException
     * @throws \think\db\exception\DbException
     * @throws \think\db\exception\ModelNotFoundException
     */
    public function checkDel($value, $rule, $data)
    {
        $order = Order::where('id',$value)->find()->toArray();

        if ($order['order_status'] != OrderEnum::STATUS_CLOSE) {
            return '订单无法删除';
        }
        //如果在售后中无法删除
        if(AfterSale::where(['order_id'=>$value,'status'=>AfterSaleEnum::STATUS_ING])->find()){
            return '订单正在售后中，无法删除';
        }

        return true;
    }

}