<?php
/**
 *  ╔═══════════════════════════════════════════════════╗
 *  ║                                                   ║
 *  ║     ██╗  ██╗   █████╗    ██████╗                  ║
 *  ║     ██║  ██║  ██╔══██╗  ██╔═══██╗                 ║
 *  ║     ███████║  ███████║  ██║   ██║                 ║
 *  ║     ██╔══██║  ██╔══██║  ██║   ██║                 ║
 *  ║     ██║  ██║  ██║  ██║  ╚██████╔╝   SNS           ║
 *  ║                                                   ║    
 *  ║                                                   ║    
 *  ║     © 2023 HaoSNS™ All Rights Reserved            ║
 *  ║     官方网站: https://www.haosns.com                *
 *  ║     本代码由赣州乐易网络科技有限公司®提供             *
 *  ║                                                    *
 *  ║   未经授权禁止复制、传播或用于其他商业目的            *
 *  ║                                                   ║
 *  ╚═══════════════════════════════════════════════════╝
 */


namespace app\api\logic\Order;


use app\common\enum\AfterSaleEnum;
use app\common\enum\DeliveryEnum;
use app\common\enum\OrderEnum;
use app\common\enum\PayEnum;
use app\common\enum\YesNoEnum;
use app\common\logic\BaseLogic;
use app\common\model\AfterSale;
use app\common\model\Order;
use app\common\service\ConfigService;

/**
 * 订单按钮
 * Class OrderBtnLogic
 * @package app\api\logic\Order
 */
class OrderBtnLogic extends BaseLogic
{

    /**
     * @notes 订单按钮状态
     * @param Order $order
     * @return array
     */
    public static function getOrderBtn(Order $order)
    {
        return [
            'pay_btn'       => self::getPayBtn($order),
            'cancel_btn'    => self::getCancelBtn($order),
            'delivery_btn'  => self::getDeliveryBtn($order),
            'confirm_btn'   => self::getConfirmBtn($order),
            'finish_btn'    => self::getFinishBtn($order),
            'refund_btn'    => self::getRefundBtn($order),
            'delete_btn'    => self::getDeletedBtn($order)
        ];
    }

    /**
     * @notes 商家端订单按钮状态
     * @param Order $order
     * @return array
     * @author lgp
     * @date 2023/2/20 17:37
     */
    public static function getBusinesseBtn(Order $order)
    {
        return [
            'editaddress_btn'       => self::getEitaddressBtn($order),
            'deliver_btn'           => self::getDeliverBtn($order),
            'delivery_btn'          => self::getDeliveryBtn($order),
            'cancel_btn'            => self::getCancelBtn($order),
            'confirm_btn'           => self::getConfirmBtn($order),
            'finish_btn'            => self::getFinishBtn($order),
            'delete_btn'            => self::getDeletedBtn($order)
        ];
    }

    /**
     * @notes 支付按钮
     * @param $order
     * @return int
     */
    public static function getPayBtn($order)
    {
        if ($order['order_status'] == OrderEnum::STATUS_WAIT_PAY && $order['pay_status'] == PayEnum::UNPAID) {
            return OrderEnum::BTN_SHOW;
        }
        return OrderEnum::BTN_HIDE;
    }


    /**
     * @notes 取消按钮
     * @param $order
     * @return int
     */
    public static function getCancelBtn($order)
    {
        $btn = OrderEnum::BTN_HIDE;
        //未支付的允许取消,订单已支付时，在允许取消的时间内并且订单未发货，允许取消
        if ($order['order_status'] == OrderEnum::STATUS_WAIT_PAY
            || $order['order_status'] == OrderEnum::STATUS_WAIT_DELIVERY
        ) {
            $btn = OrderEnum::BTN_SHOW;
        }
        if ($order['order_status'] == OrderEnum::STATUS_WAIT_DELIVERY) {
            $ableCancelOrder = ConfigService::get('transaction', 'cancel_unshipped_orders');
            if ($ableCancelOrder == YesNoEnum::NO) {
                $btn = OrderEnum::BTN_HIDE;
            }
            $configTime = ConfigService::get('transaction', 'cancel_unshipped_orders_times');
            $ableCancelTime = strtotime($order['create_time']) + ($configTime * 60);
            if (time() > $ableCancelTime) {
                $btn = OrderEnum::BTN_HIDE;
            }
        }
        return $btn;
    }


    /**
     * @notes 物流按钮
     * @param $order
     * @return int
     */
    public static function getDeliveryBtn($order)
    {
        $btn = OrderEnum::BTN_HIDE;
        if ($order['order_status'] >= OrderEnum::STATUS_WAIT_RECEIVE
            && $order['pay_status'] == PayEnum::ISPAID
            && $order['express_status'] == YesNoEnum::YES
        ) {
            $btn = OrderEnum::BTN_SHOW;
        }
        return $btn;
    }


    /**
     * @notes 确认收货按钮
     * @param $order
     * @return int
     */
    public static function getConfirmBtn($order)
    {
        $btn = OrderEnum::BTN_HIDE;
        if ($order['order_status'] == OrderEnum::STATUS_WAIT_RECEIVE && $order['express_status'] == YesNoEnum::YES) {
            $btn = OrderEnum::BTN_SHOW;
        }

        return $btn;
    }

    /**
     * @notes 完成按钮
     * @param $order
     * @return int
     */
    public static function getFinishBtn($order)
    {
        $btn = OrderEnum::BTN_HIDE;
        if ($order['order_status'] == OrderEnum::STATUS_FINISH) {
            $btn = OrderEnum::BTN_SHOW;
        }
        return $btn;
    }


    /**
     * @notes 申请退款按钮
     * @param $order
     * @return int
     */
    public static function getRefundBtn($order)
    {
        $btn = OrderEnum::BTN_HIDE;
        //订单已完成、在售后期内。未申请退款、
        if ($order['order_status'] == OrderEnum::STATUS_FINISH && $order['after_sale_deadline'] > time()) {
            $checkRefund = AfterSale::where([
                ['order_id', '=', $order['id']],
                ['status', 'in', [AfterSaleEnum::STATUS_ING, AfterSaleEnum::STATUS_SUCCESS]]
            ])->order(['id' => 'desc'])->findOrEmpty();

            if ($checkRefund->isEmpty()) {
                $btn = OrderEnum::BTN_SHOW;
            }
        }
        return $btn;
    }


    /**
     * @notes 删除订单按钮
     * @param $order
     * @return int
     */
    public static function getDeletedBtn($order)
    {
        $btn = OrderEnum::BTN_HIDE;
        //订单已关闭
        if ($order['order_status'] == OrderEnum::STATUS_CLOSE) {
            $btn = OrderEnum::BTN_SHOW;
        }
        return $btn;
    }

    /**
     * @notes 修改地址按钮
     * @param $order
     * @return int
     * @author lgp
     * @date 2023/2/17 15:36
     */
    public static function getEitaddressBtn($order)
    {
        $btn = OrderEnum::BTN_HIDE;
        if(DeliveryEnum::EXPRESS_DELIVERY == $order['delivery_type'] && (OrderEnum::STATUS_WAIT_PAY == $order['order_status'] || OrderEnum::STATUS_WAIT_DELIVERY == $order['order_status'])) {
            $btn = OrderEnum::BTN_SHOW;
        }
        return $btn;
    }

    /**
     * @notes 去发货按钮
     * @param $order
     * @return int
     * @author lgp
     * @date 2023/2/17 15:38
     */
    public static function getDeliverBtn($order)
    {
        $btn = OrderEnum::BTN_HIDE;
        if(OrderEnum::STATUS_WAIT_DELIVERY == $order['order_status']) {
            $btn = OrderEnum::BTN_SHOW;
        }
        return $btn;
    }

}