<?php
/**
 *  ╔═══════════════════════════════════════════════════╗
 *  ║                                                   ║
 *  ║     ██╗  ██╗   █████╗    ██████╗                  ║
 *  ║     ██║  ██║  ██╔══██╗  ██╔═══██╗                 ║
 *  ║     ███████║  ███████║  ██║   ██║                 ║
 *  ║     ██╔══██║  ██╔══██║  ██║   ██║                 ║
 *  ║     ██║  ██║  ██║  ██║  ╚██████╔╝   SNS           ║
 *  ║                                                   ║    
 *  ║                                                   ║    
 *  ║     © 2023 HaoSNS™ All Rights Reserved            ║
 *  ║     官方网站: https://www.haosns.com                *
 *  ║     本代码由赣州乐易网络科技有限公司®提供             *
 *  ║                                                    *
 *  ║   未经授权禁止复制、传播或用于其他商业目的            *
 *  ║                                                   ║
 *  ╚═══════════════════════════════════════════════════╝
 */
namespace app\api\validate\twoshop;


use app\api\model\Goods;
use app\common\model\GoodsSupplier;
use app\common\model\GoodsSupplierCategory;
use app\common\validate\BaseValidate;

class GoodsSupplierValidate extends BaseValidate
{
    protected $rule = [
        'id' => 'require|checkId|checkDel',
        'code' => 'require|checkCode',
        'name' => 'require|checkName',
        'supplier_category_id' => 'require|checkCategory',
        'mobile' => 'mobile',
        'email' => 'email',
        'sort' => 'number|max:5',
    ];

    protected $message = [
        'code.require' => '供应商编码不能为空',
        'name.require' => '供应商名称不能为空',
        'supplier_category_id.require' => '供应商分类不能为空',
        'mobile.mobile' => '无效的手机号码',
        'email.email' => '邮箱格式不正确',
        'sort.number' => '排序必须是纯数字',
        'sort.max' => '排序最大不能超过五位数',
    ];

    public function sceneAdd()
    {
        return $this->only(['code','name','supplier_category_id','mobile','email','sort']);
    }

    public function sceneDel()
    {
        return $this->only(['id']);
    }

    public function sceneEdit()
    {
        return $this->only(['id','code','name','supplier_category_id','mobile','email','sort'])
            ->remove('id','checkDel');
    }

    public function sceneDetail()
    {
        return $this->only(['id'])
            ->remove('id','checkDel');
    }

    /**
     * @notes 检查供应商编码是否已存在
     * @param $value
     * @param $rule
     * @param $data
     * @return bool|string
     * @throws \think\db\exception\DataNotFoundException
     * @throws \think\db\exception\DbException
     * @throws \think\db\exception\ModelNotFoundException
     */
    public function checkCode($value,$rule,$data)
    {
        $where[] = ['code', '=', $value];
        // 编辑的情况，要排除自身ID
        if (isset($data['id'])) {
            $where[] = ['id', '<>', $data['id']];
        }
        $result = GoodsSupplier::where($where)->select()->toArray();
        if ($result) {
            return '供应商编码已存在';
        }
        return true;
    }

    /**
     * @notes 检查供应商名称是否已存在
     * @param $value
     * @param $rule
     * @param $data
     * @return bool|string
     * @throws \think\db\exception\DataNotFoundException
     * @throws \think\db\exception\DbException
     * @throws \think\db\exception\ModelNotFoundException
     */
    public function checkName($value,$rule,$data)
    {
        $where[] = ['name', '=', $value];
        // 编辑的情况，要排除自身ID
        if (isset($data['id'])) {
            $where[] = ['id', '<>', $data['id']];
        }
        $result = GoodsSupplier::where($where)->select()->toArray();
        if ($result) {
            return '该供应商名称已存在';
        }
        return true;
    }

    /**
     * @notes 检查供应商分类是否存在
     * @param $value
     * @param $rule
     * @param $data
     * @return bool|string
     */
    public function checkCategory($value,$rule,$data)
    {
        $result = GoodsSupplierCategory::findOrEmpty($value);
        if ($result->isEmpty()) {
            return '供应商分类不存在';
        }
        return true;
    }

    /**
     * @notes 检查供应商id是否存在
     * @param $value
     * @param $rule
     * @param $data
     * @return bool|string
     */
    public function checkId($value,$rule,$data)
    {
        $result = GoodsSupplier::findOrEmpty($value);
        if ($result->isEmpty()) {
            return '供应商不存在';
        }
        return true;
    }

    /**
     * @notes 检查供应商是否能删除
     * @param $value
     * @param $rule
     * @param $data
     * @return bool|string
     * @throws \think\db\exception\DataNotFoundException
     * @throws \think\db\exception\DbException
     * @throws \think\db\exception\ModelNotFoundException
     * @throws \think\exception\DbException
     */
    public function checkDel($value,$rule,$data)
    {
        $result = GoodsSupplier::hasWhere('goods',['supplier_id'=>$value])->select()->toArray();
        if (!empty($result)) {
            return '供应商正在使用中，无法删除';
        }
        return true;
    }
}