<?php
/**
 *  ╔═══════════════════════════════════════════════════╗
 *  ║                                                   ║
 *  ║     ██╗  ██╗   █████╗    ██████╗                  ║
 *  ║     ██║  ██║  ██╔══██╗  ██╔═══██╗                 ║
 *  ║     ███████║  ███████║  ██║   ██║                 ║
 *  ║     ██╔══██║  ██╔══██║  ██║   ██║                 ║
 *  ║     ██║  ██║  ██║  ██║  ╚██████╔╝   SNS           ║
 *  ║                                                   ║    
 *  ║                                                   ║    
 *  ║     © 2023 HaoSNS™ All Rights Reserved            ║
 *  ║     官方网站: https://www.haosns.com                *
 *  ║     本代码由赣州乐易网络科技有限公司®提供             *
 *  ║                                                    *
 *  ║   未经授权禁止复制、传播或用于其他商业目的            *
 *  ║                                                   ║
 *  ╚═══════════════════════════════════════════════════╝
 */


namespace app\adminapi\lists\twoshop;

use app\adminapi\lists\BaseAdminDataLists;
use app\common\enum\AfterSaleEnum;
use app\common\lists\ListsExcelInterface;
use app\common\lists\ListsExtendInterface;
use app\common\model\TwoAfterSale;
use app\common\service\FileService;

/**
 * 售后列表类
 * Class AfterSaleLists
 * @package app\adminapi\lists\after_sale
 */
class AfterSaleLists extends BaseAdminDataLists implements ListsExtendInterface,ListsExcelInterface
{
    /**
     * @notes 导出字段
     * @return string[]
     */
    public function setExcelFields(): array
    {
        return [
            'sn' => '售后单号',
            'nickname' => '用户昵称',
            'order_sn' => '订单编号',
            'refund_type_desc' => '售后类型',
            'refund_method_desc' => '售后方式',
            'refund_total_amount' => '售后金额',
            'sub_status_desc' => '售后状态',
            'create_time' => '申请时间',
        ];
    }

    /**
     * @notes 导出表名
     * @return string
     */
    public function setFileName(): string
    {
        return '售后列表';
    }

    /**
     * @notes 统计信息
     * @return mixed|void
     */
    public function extend()
    {
        $all = TwoAfterSale::count();
        $ing = TwoAfterSale::where('status', AfterSaleEnum::STATUS_ING)->count();
        $success = TwoAfterSale::where('status', AfterSaleEnum::STATUS_SUCCESS)->count();
        $fail = TwoAfterSale::where('status', AfterSaleEnum::STATUS_FAIL)->count();
        return [
            'all' => $all,
            'ing' => $ing,
            'success' => $success,
            'fail' => $fail
        ];
    }

    /**
     * @notes 设置搜索
     */
    public function setSearch()
    {
        // 售后编号
        if(isset($this->params['after_sale_sn']) && $this->params['after_sale_sn']) {
            $this->searchWhere[] = ['as.sn', 'like', '%'.$this->params['after_sale_sn'].'%'];
        }
        // 订单编号
        if(isset($this->params['order_sn']) && $this->params['order_sn']) {
            $this->searchWhere[] = ['o.sn', 'like', '%'.$this->params['order_sn'].'%'];
        }
        // 用户信息
        if(isset($this->params['user_info']) && $this->params['user_info']) {
            $this->searchWhere[] = ['u.sn|u.nickname|u.mobile', 'like', '%'.$this->params['user_info'].'%'];
        }
        // 售后类型
        if(isset($this->params['refund_type']) && $this->params['refund_type']) {
            $this->searchWhere[] = ['as.refund_type', '=', $this->params['refund_type']];
        }
        // 售后方式
        if(isset($this->params['refund_method']) && $this->params['refund_method']) {
            $this->searchWhere[] = ['as.refund_method', '=', $this->params['refund_method']];
        }
        // 申请时间
        if(isset($this->params['start_time']) && $this->params['start_time']) {
            $this->searchWhere[] = ['as.create_time', '>=', strtotime($this->params['start_time'])];
        }
        if(isset($this->params['end_time']) && $this->params['end_time']) {
            $this->searchWhere[] = ['as.create_time', '<=', strtotime($this->params['end_time'])];
        }
        // 售后主状态
        if(isset($this->params['status']) && $this->params['status']) {
            $this->searchWhere[] = ['as.status', '=', $this->params['status']];
        }
        // 快递单号
        if(isset($this->params['invoice_no']) && $this->params['invoice_no']) {
            $this->searchWhere[] = ['as.invoice_no', 'like', '%'.$this->params['invoice_no'].'%'];
        }
    }

    /**
     * @notes 售后列表
     * @return array
     * @throws \think\db\exception\DataNotFoundException
     * @throws \think\db\exception\DbException
     * @throws \think\db\exception\ModelNotFoundException
     */
    public function lists() : array
    {
        // 设置搜索
        $this->setSearch();

        $field = 'as.id,as.user_id,as.order_id,as.sn,as.refund_type,as.refund_method,as.refund_total_amount,as.status,as.sub_status,as.create_time';
        $field .= ',as.refund_type as refund_type_desc,as.refund_method as refund_method_desc,as.status as status_desc,as.sub_status as sub_status_desc';
        $field .= ',o.sn as order_sn';
        $field .= ',u.nickname,u.avatar,u.sn as user_sn';

        $lists = TwoAfterSale::with(['after_sale_goods'])
            ->alias('as')
            ->leftJoin('two_order o', 'o.id = as.order_id')
            ->leftJoin('user u', 'u.id = as.user_id')
            ->field($field)
            ->where($this->searchWhere)
            ->withSearch(['goods_info'], $this->params)
            ->limit($this->limitOffset, $this->limitLength)
            ->order('id', 'desc')
            ->select()
            ->toArray();

        foreach($lists as &$item) {
            $item['avatar'] = FileService::getFileUrl($item['avatar']);
        }
        
        return $lists;
    }

    /**
     * @notes 售后记录数
     * @return int
     */
    public function  count() : int
    {
        // 设置搜索
        $this->setSearch();

        $count = TwoAfterSale::alias('as')
            ->leftJoin('order o', 'o.id = as.order_id')
            ->leftJoin('user u', 'u.id = as.user_id')
            ->where($this->searchWhere)->count();

        return $count;
    }
}