<?php
/**
 *  ╔═══════════════════════════════════════════════════╗
 *  ║                                                   ║
 *  ║     ██╗  ██╗   █████╗    ██████╗                  ║
 *  ║     ██║  ██║  ██╔══██╗  ██╔═══██╗                 ║
 *  ║     ███████║  ███████║  ██║   ██║                 ║
 *  ║     ██╔══██║  ██╔══██║  ██║   ██║                 ║
 *  ║     ██║  ██║  ██║  ██║  ╚██████╔╝   SNS           ║
 *  ║                                                   ║    
 *  ║                                                   ║    
 *  ║     © 2023 HaoSNS™ All Rights Reserved            ║
 *  ║     官方网站: https://www.haosns.com                *
 *  ║     本代码由赣州乐易网络科技有限公司®提供             *
 *  ║                                                    *
 *  ║   未经授权禁止复制、传播或用于其他商业目的            *
 *  ║                                                   ║
 *  ╚═══════════════════════════════════════════════════╝
 */

namespace app\adminapi\lists\goods;

use app\adminapi\{
    lists\BaseAdminDataLists,
    logic\goods\GoodsCategoryLogic,
};
use app\common\{enum\GoodsEnum,
    lists\ListsExcelInterface,
    lists\ListsExtendInterface,
    model\Goods};


/**
 * 商品列表接口
 * Class GoodsLists
 * @package app\adminapi\lists\goods
 */
class GoodsLists extends BaseAdminDataLists implements ListsExtendInterface,ListsExcelInterface
{

    /**
     * @notes 搜索条件
     * @return array
     */
    public function setSearch(): array
    {

        return array_intersect(array_keys($this->params),['type','keyword','category_id','supplier_id','goods_type','activity_type']);
    }

    /**
     * @notes 统计信息
     * @return array
     * @throws \think\db\exception\DataNotFoundException
     * @throws \think\db\exception\DbException
     * @throws \think\db\exception\ModelNotFoundException
     */
    public function extend(): array
    {
        $statistics = (new Goods())
            ->withSearch(array_diff(['type'],$this->setSearch()), $this->params)
            ->field('count(id) as all_count,
                    IFNULL(sum(if(status = 1,1,0)),0) as sales_count,
                    IFNULL(sum(if(status = 1 and stock_warning > 0 and total_stock > 0 and stock_warning > total_stock,1,0)),0) as warning_count,
                    IFNULL(sum(if(status = 1 and total_stock = 0,1,0)),0) as sellout_count,
                    IFNULL(sum(if(status = 0,1,0)),0) as storage_count')
            ->select()->toArray();
        
        return array_shift($statistics);
    }


    /**
     * @notes 商品列表
     * @return array
     * @throws \think\db\exception\DataNotFoundException
     * @throws \think\db\exception\DbException
     * @throws \think\db\exception\ModelNotFoundException
     */
    public function lists(): array
    {

        $list = (new Goods())
            ->withSearch($this->setSearch(), $this->params)
            ->with(['goods_category_index'])
            ->limit($this->limitOffset, $this->limitLength)
            ->field('id,name,code,image,min_price,max_price,total_stock,virtual_sales_num+sales_num as sales_num,virtual_click_num+click_num as click_num,status,sort,spec_type,create_time')
            ->order('id desc')
            ->select()
            ->toArray();
        //获取参与活动的商品
        $goodsIds = array_column($list,'id');

        //获取全部商品分类
        $categoryList = GoodsCategoryLogic::getCategoryNameLists();

        foreach ($list as $goodsKey => $goodsVal) {
            $list[$goodsKey]['status_desc'] = GoodsEnum::getStatusDesc($goodsVal['status']);
            //商品价格
            $list[$goodsKey]['price'] = '¥' . $goodsVal['min_price'];
            if ($goodsVal['min_price'] != $goodsVal['max_price']) {
                $list[$goodsKey]['price'] = '¥' . $goodsVal['min_price'] . '~' . '¥' . $goodsVal['max_price'];
            }
            //商品分类
            $categoryName = array_map(function ($categoryIndex) use ($categoryList) {
                return $categoryList[$categoryIndex['category_id']]['category_name'] ?? '';
            }, $goodsVal['goods_category_index']);

            $list[$goodsKey]['category_name'] = $categoryName;
            unset($list[$goodsKey]['goods_category_index']);
            
        }
        return $list;

    }


    /**
     * @notes 商品总数
     * @return int
     */
    public function count(): int
    {
        return (new Goods())
            ->withSearch($this->setSearch(), $this->params)
            ->count();
    }

    /**
     * @notes 设置excel表名
     * @return string
     */
    public function setFileName(): string
    {
        return '商品列表';
    }

    /**
     * @notes 设置导出字段
     * @return array
     */
    public function setExcelFields(): array
    {
        return [
            'code'          => '商品编号',
            'name'          => '商品名称',
            'price'         => '价格',
            'total_stock'   => '库存',
            'sales_num'     => '销量',
            'click_num'     => '浏览量',
            'status_desc'   => '销售状态',
            'sort'          => '排序',
            'create_time'   => '创建时间',
        ];
    }


}