<?php
/**
 *  ╔═══════════════════════════════════════════════════╗
 *  ║                                                   ║
 *  ║     ██╗  ██╗   █████╗    ██████╗                  ║
 *  ║     ██║  ██║  ██╔══██╗  ██╔═══██╗                 ║
 *  ║     ███████║  ███████║  ██║   ██║                 ║
 *  ║     ██╔══██║  ██╔══██║  ██║   ██║                 ║
 *  ║     ██║  ██║  ██║  ██║  ╚██████╔╝   SNS           ║
 *  ║                                                   ║    
 *  ║                                                   ║    
 *  ║     © 2023 HaoSNS™ All Rights Reserved            ║
 *  ║     官方网站: https://www.haosns.com                *
 *  ║     本代码由赣州乐易网络科技有限公司®提供             *
 *  ║                                                    *
 *  ║   未经授权禁止复制、传播或用于其他商业目的            *
 *  ║                                                   ║
 *  ╚═══════════════════════════════════════════════════╝
 */


namespace app\adminapi\lists;

use app\common\enum\AccountLogEnum;
use app\common\lists\ListsExcelInterface;
use app\common\lists\ListsSearchInterface;
use app\common\model\AccountLog;

/**
 * 账记流水列表
 * Class AccountLogLists
 * @package app\adminapi\lists
 */
class AccountLogLists extends BaseAdminDataLists implements ListsSearchInterface,ListsExcelInterface
{
    /**
     * @notes 导出表名
     * @return string
     */
    public function setFileName(): string
    {
        return '余额明细';
    }

    /**
     * @notes 导出字段
     * @return array
     */
    public function setExcelFields(): array
    {
        return [
            'nickname' => '用户昵称',
            'sn' => '用户编号',
            'mobile' => '手机号码',
            'change_amount' => '变动金额',
            'left_amount' => '剩余金额',
            'change_type_desc' => '变动类型',
            'association_sn' => '来源单号',
            'create_time' => '记录时间',
        ];
    }

    /**
     * @notes 设置搜索
     * @return array
     */
    public function setSearch(): array
    {
        return [
            '=' => ['u.sn', 'u.mobile', 'al.change_type'],
            '%like%' => ['u.nickname'],
            'between_time' => 'al.create_time'
        ];
    }

    /**
     * @notes 自定义搜索
     */
    public function queryWhere()
    {
        // 可提现余额类型
        if(isset($this->params['type']) && $this->params['type'] == 'bw') {
            $this->searchWhere[] = ['change_type', 'in', AccountLogEnum::getBwChangeType()];
        }

        // 不可提现余额类型
        if(isset($this->params['type']) && $this->params['type'] == 'bnw') {
            $this->searchWhere[] = ['change_type', 'in', AccountLogEnum::getBnwChangeType()];
        }

    }

    /**
     * @notes 账户流水列表
     * @return array
     */
    public function lists(): array
    {
        $this->queryWhere();
        $field = 'u.nickname,u.sn,u.mobile,al.action,al.change_amount,al.left_amount,al.change_type,al.association_sn,al.create_time';
        $lists = AccountLog::alias('al')
            ->leftJoin('user u', 'u.id = al.user_id')
            ->field($field)
            ->where($this->searchWhere)
            ->order('al.id', 'desc')
            ->limit($this->limitOffset, $this->limitLength)
            ->select()
            ->toArray();

        foreach($lists as &$item) {
            $item['change_type_desc'] = AccountLogEnum::getChangeTypeDesc( $item['change_type']);
            $symbol = $item['action'] == AccountLogEnum::INC ? '+' : '-';
            // 成长值转整型
            $item['change_amount'] =  in_array($item['change_type'], AccountLogEnum::getGrowthChangeType()) ? (int)$item['change_amount'] : $item['change_amount'];
            $item['left_amount']   =   in_array($item['change_type'], AccountLogEnum::getGrowthChangeType()) ? (int)$item['left_amount'] : $item['left_amount'];
            $item['change_amount'] = $symbol . $item['change_amount'];
        }

        return $lists;
    }

    /**
     * @notes 账户流水数量
     * @return int
     */
    public function count(): int
    {
        $this->queryWhere();
        $count = AccountLog::alias('al')
            ->leftJoin('user u', 'u.id = al.user_id')
            ->where($this->searchWhere)
            ->count();

        return $count;
    }
}