<?php
/**
 *  ╔═══════════════════════════════════════════════════╗
 *  ║                                                   ║
 *  ║     ██╗  ██╗   █████╗    ██████╗                  ║
 *  ║     ██║  ██║  ██╔══██╗  ██╔═══██╗                 ║
 *  ║     ███████║  ███████║  ██║   ██║                 ║
 *  ║     ██╔══██║  ██╔══██║  ██║   ██║                 ║
 *  ║     ██║  ██║  ██║  ██║  ╚██████╔╝   SNS           ║
 *  ║                                                   ║    
 *  ║                                                   ║    
 *  ║     © 2023 HaoSNS™ All Rights Reserved            ║
 *  ║     官方网站: https://www.haosns.com                *
 *  ║     本代码由赣州乐易网络科技有限公司®提供             *
 *  ║                                                    *
 *  ║   未经授权禁止复制、传播或用于其他商业目的            *
 *  ║                                                   ║
 *  ╚═══════════════════════════════════════════════════╝
 */


namespace app\adminapi\validate\sign;

use app\common\enum\YesNoEnum;
use app\common\model\SignDaily;
use app\common\validate\BaseValidate;

/**
 * 签到验证器
 * Class SignValidate
 * @package app\adminapi\validate\sign
 */
class SignValidate extends BaseValidate
{
    protected $rule = [
        'is_open' => 'require',
        'daily' => 'require|array|checkDaily',
        'remark' => 'require',
        'days' => 'require|egt:2|checkDays',
        'money_status' => 'require|in:0,1',
        'money' => 'require|gt:0',
        'id' => 'require'
    ];

    protected $message = [
        'is_open.require' => '请选择签到状态',
        'daily.require' => '请填写每日签到规则',
        'daily.array' => '每日签到规则格式须为数组',
        'remark.require' => '请填写签到说明',
        'days.require' => '请填写连续签到天数',
        'days.egt' => '连续签到天数须大于或等于2天',
        'money_status.require' => '请选择是否赠送金币',
        'money_status.in' => '赠送金币状态错误',
        'money.require' => '请填写赠送金币数量',
        'money.gt' => '赠送金币数量须大于0',
        'id.require' => '参数缺失',
    ];

    /**
     * @notes 设置签到规则场景
     * @param $value
     * @param $rule
     * @param $data
     * @return SignValidate
     */
    public function sceneSetConfig()
    {
        return $this->only(['is_open', 'daily', 'remark']);
    }

    /**
     * @notes 添加连续签到规则场景
     * @return SignValidate
     */
    public function sceneAdd()
    {
        return $this->only(['days', 'money_status', 'money']);
    }

    /**
     * @notes 编辑连续签到规则场景
     * @return SignValidate
     */
    public function sceneEdit()
    {
        return $this->only(['id', 'days', 'money_status', 'money']);
    }

    /**
     * @notes 查看连续签到规则详情场景
     * @return SignValidate
     */
    public function sceneDetail()
    {
        return $this->only(['id']);
    }

    /**
     * @notes 删除连续签到规则场景
     * @return SignValidate
     */
    public function sceneDelete()
    {
        return $this->only(['id']);
    }

    /**
     * @notes 校验每日签到参数
     * @param $value
     * @return bool|string
     */
    public function checkDaily($value)
    {
        if(
            !isset($value['money_status']) ||
            !isset($value['money'])
        ) {
            return '每日签到参数缺失';
        }
        if(
            !in_array($value['money_status'], [YesNoEnum::YES, YesNoEnum::NO]) ||
            $value['money'] < 0
        ) {
            return '每日签到参数不符合要求';
        }
        return true;
    }

    /**
     * @notes 校验连续签到天数
     * @param $value
     * @param $rule
     * @param $data
     * @return bool|string
     */
    public function checkDays($value, $rule, $data)
    {
        $where[] = ['days', '=', $value];
        if(isset($data['id'])) {
            $where[] = ['id', '<>', $data['id']];
        }
        $signDaily = SignDaily::where($where)->findOrEmpty();
        if(!$signDaily->isEmpty()) {
            return '已有相同的连续签到规则';
        }
        return true;
    }
}